<?php
/**
 * Bot Health Check Cron Job
 * Monitors bot status and restarts if needed
 */

require_once __DIR__ . '/../config.php';
require_once __DIR__ . '/../includes/functions.php';

// Set execution time limit
set_time_limit(300); // 5 minutes

// Log file
$logFile = __DIR__ . '/../logs/bot_health.log';

// Ensure logs directory exists
if (!is_dir(__DIR__ . '/../logs')) {
    mkdir(__DIR__ . '/../logs', 0755, true);
}

function logMessage($message) {
    global $logFile;
    $timestamp = date('Y-m-d H:i:s');
    $logEntry = "[$timestamp] $message" . PHP_EOL;
    file_put_contents($logFile, $logEntry, FILE_APPEND | LOCK_EX);
    echo $logEntry;
}

logMessage("=== BOT HEALTH CHECK STARTED ===");

try {
    // Check if PHP bot is running
    $phpBotLockFile = __DIR__ . '/../bot/php_bot.lock';
    $phpBotRunning = file_exists($phpBotLockFile);
    
    if ($phpBotRunning) {
        $lockTime = filemtime($phpBotLockFile);
        $lockAge = time() - $lockTime;
        
        // If lock file is older than 30 minutes, consider it stale
        if ($lockAge > 1800) {
            logMessage("WARNING: PHP bot lock file is stale (age: {$lockAge}s). Removing stale lock.");
            unlink($phpBotLockFile);
            $phpBotRunning = false;
        } else {
            logMessage("INFO: PHP bot is running (lock age: {$lockAge}s)");
        }
    } else {
        logMessage("INFO: PHP bot is not running");
    }
    
    // Check if Python bot is running
    $pythonBotLockFile = __DIR__ . '/../bot/python_bot.lock';
    $pythonBotRunning = file_exists($pythonBotLockFile);
    
    if ($pythonBotRunning) {
        $lockTime = filemtime($pythonBotLockFile);
        $lockAge = time() - $lockTime;
        
        // If lock file is older than 30 minutes, consider it stale
        if ($lockAge > 1800) {
            logMessage("WARNING: Python bot lock file is stale (age: {$lockAge}s). Removing stale lock.");
            unlink($pythonBotLockFile);
            $pythonBotRunning = false;
        } else {
            logMessage("INFO: Python bot is running (lock age: {$lockAge}s)");
        }
    } else {
        logMessage("INFO: Python bot is not running");
    }
    
    // Check database connection
    try {
        $pdo = getDB();
        $stmt = $pdo->query("SELECT COUNT(*) as count FROM students WHERE email_verified = 1");
        $result = $stmt->fetch();
        $verifiedStudents = $result['count'];
        logMessage("INFO: Database connection OK. Verified students: $verifiedStudents");
    } catch (Exception $e) {
        logMessage("ERROR: Database connection failed: " . $e->getMessage());
    }
    
    // Check Moodle API connection
    try {
        $moodleInfo = callMoodleAPI('core_webservice_get_site_info');
        if ($moodleInfo && isset($moodleInfo['sitename'])) {
            logMessage("INFO: Moodle API connection OK. Site: " . $moodleInfo['sitename']);
        } else {
            logMessage("WARNING: Moodle API connection failed or returned invalid data");
        }
    } catch (Exception $e) {
        logMessage("ERROR: Moodle API connection failed: " . $e->getMessage());
    }
    
    // Check disk space
    $diskFree = disk_free_space(__DIR__ . '/../');
    $diskTotal = disk_total_space(__DIR__ . '/../');
    $diskUsed = $diskTotal - $diskFree;
    $diskPercent = round(($diskUsed / $diskTotal) * 100, 2);
    
    if ($diskPercent > 90) {
        logMessage("WARNING: Disk usage is high: {$diskPercent}%");
    } else {
        logMessage("INFO: Disk usage: {$diskPercent}%");
    }
    
    // Check if we need to restart bots
    $restartNeeded = false;
    
    // Check for stuck processes
    $processes = [];
    if (function_exists('exec')) {
        exec('ps aux | grep -E "(php_bot|moodle_sync_bot)" | grep -v grep', $processes);
    }
    
    if (count($processes) > 0) {
        logMessage("INFO: Found " . count($processes) . " bot processes running");
    } else {
        logMessage("INFO: No bot processes found running");
    }
    
    logMessage("=== BOT HEALTH CHECK COMPLETED ===");
    
} catch (Exception $e) {
    logMessage("ERROR: Bot health check failed: " . $e->getMessage());
    exit(1);
}

logMessage("Bot health check completed successfully");
?>
