<?php
/**
 * Security Headers for Skolo-Kine Learning Platform
 */

// Prevent direct access
if (!defined('SECURITY_HEADERS_LOADED')) {
    define('SECURITY_HEADERS_LOADED', true);
    
    // Set security headers only if not running from command line
    if (php_sapi_name() !== 'cli') {
        setSecurityHeaders();
    }
}

/**
 * Set comprehensive security headers
 */
function setSecurityHeaders() {
    // Prevent MIME type sniffing
    header('X-Content-Type-Options: nosniff');
    
    // Prevent clickjacking
    header('X-Frame-Options: DENY');
    
    // XSS Protection
    header('X-XSS-Protection: 1; mode=block');
    
    // Referrer Policy
    header('Referrer-Policy: strict-origin-when-cross-origin');
    
    // Content Security Policy
    $csp = "default-src 'self'; " .
           "script-src 'self' 'unsafe-inline' 'unsafe-eval' https://cdn.jsdelivr.net https://cdnjs.cloudflare.com; " .
           "style-src 'self' 'unsafe-inline' https://cdn.jsdelivr.net https://fonts.googleapis.com; " .
           "font-src 'self' https://fonts.gstatic.com https://cdn.jsdelivr.net; " .
           "img-src 'self' data: https:; " .
           "connect-src 'self' https://skoloi.melanegroup.com https://sandbox.momodeveloper.mtn.com; " .
           "frame-ancestors 'none'; " .
           "base-uri 'self'; " .
           "form-action 'self'; " .
           "object-src 'none'; " .
           "upgrade-insecure-requests";
    
    header("Content-Security-Policy: $csp");
    
    // Strict Transport Security (HTTPS only)
    if (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on') {
        header('Strict-Transport-Security: max-age=31536000; includeSubDomains; preload');
    }
    
    // Permissions Policy (formerly Feature Policy)
    $permissions = "geolocation=(), " .
                   "microphone=(), " .
                   "camera=(), " .
                   "payment=(), " .
                   "usb=(), " .
                   "magnetometer=(), " .
                   "gyroscope=(), " .
                   "speaker=(), " .
                   "vibrate=(), " .
                   "fullscreen=(self), " .
                   "sync-xhr=()";
    
    header("Permissions-Policy: $permissions");
    
    // Additional security headers
    header('X-Permitted-Cross-Domain-Policies: none');
    header('Cross-Origin-Embedder-Policy: require-corp');
    header('Cross-Origin-Opener-Policy: same-origin');
    header('Cross-Origin-Resource-Policy: same-origin');
    
    // Remove server information
    header_remove('X-Powered-By');
    header_remove('Server');
}

/**
 * Set CSRF protection headers
 */
function setCSRFHeaders() {
    // Prevent CSRF by setting SameSite cookie attribute
    if (session_status() === PHP_SESSION_ACTIVE) {
        $params = session_get_cookie_params();
        session_set_cookie_params([
            'lifetime' => $params['lifetime'],
            'path' => $params['path'],
            'domain' => $params['domain'],
            'secure' => isset($_SERVER['HTTPS']),
            'httponly' => true,
            'samesite' => 'Strict'
        ]);
    }
}

/**
 * Validate and sanitize input
 */
function sanitizeInput($input, $type = 'string') {
    switch ($type) {
        case 'email':
            return filter_var(trim($input), FILTER_SANITIZE_EMAIL);
        case 'int':
            return filter_var($input, FILTER_SANITIZE_NUMBER_INT);
        case 'float':
            return filter_var($input, FILTER_SANITIZE_NUMBER_FLOAT, FILTER_FLAG_ALLOW_FRACTION);
        case 'url':
            return filter_var(trim($input), FILTER_SANITIZE_URL);
        case 'string':
        default:
            return htmlspecialchars(trim($input), ENT_QUOTES, 'UTF-8');
    }
}

/**
 * Validate file upload
 */
function validateFileUpload($file, $allowed_types = [], $max_size = 5242880) {
    $errors = [];
    
    // Check if file was uploaded
    if (!isset($file['tmp_name']) || !is_uploaded_file($file['tmp_name'])) {
        $errors[] = 'No file uploaded';
        return ['valid' => false, 'errors' => $errors];
    }
    
    // Check file size
    if ($file['size'] > $max_size) {
        $errors[] = 'File too large. Maximum size: ' . formatBytes($max_size);
    }
    
    // Check file type
    if (!empty($allowed_types)) {
        $file_type = mime_content_type($file['tmp_name']);
        if (!in_array($file_type, $allowed_types)) {
            $errors[] = 'Invalid file type. Allowed: ' . implode(', ', $allowed_types);
        }
    }
    
    // Check for malicious content
    $content = file_get_contents($file['tmp_name']);
    if (strpos($content, '<?php') !== false || strpos($content, '<script') !== false) {
        $errors[] = 'File contains potentially malicious content';
    }
    
    return [
        'valid' => empty($errors),
        'errors' => $errors
    ];
}

/**
 * Rate limiting
 */
function checkRateLimit($identifier, $max_attempts = 5, $time_window = 300) {
    $cache_file = sys_get_temp_dir() . '/rate_limit_' . md5($identifier) . '.json';
    
    $data = [];
    if (file_exists($cache_file)) {
        $data = json_decode(file_get_contents($cache_file), true) ?: [];
    }
    
    $now = time();
    $attempts = array_filter($data, function($timestamp) use ($now, $time_window) {
        return ($now - $timestamp) < $time_window;
    });
    
    if (count($attempts) >= $max_attempts) {
        return false; // Rate limit exceeded
    }
    
    // Add current attempt
    $attempts[] = $now;
    file_put_contents($cache_file, json_encode($attempts));
    
    return true;
}

/**
 * Generate secure random token
 */
function generateSecureToken($length = 32) {
    return bin2hex(random_bytes($length));
}

/**
 * Hash password securely
 */
function hashPassword($password) {
    return password_hash($password, PASSWORD_ARGON2ID, [
        'memory_cost' => 65536, // 64 MB
        'time_cost' => 4,        // 4 iterations
        'threads' => 3           // 3 threads
    ]);
}

/**
 * Verify password
 */
function verifyPassword($password, $hash) {
    return password_verify($password, $hash);
}

/**
 * Log security events
 */
function logSecurityEvent($event, $details = '', $severity = 'INFO') {
    try {
        $pdo = getDB();
        $stmt = $pdo->prepare("
            INSERT INTO security_logs (event, details, severity, ip_address, user_agent, created_at) 
            VALUES (?, ?, ?, ?, ?, NOW())
        ");
        $stmt->execute([
            $event,
            $details,
            $severity,
            $_SERVER['REMOTE_ADDR'] ?? 'unknown',
            $_SERVER['HTTP_USER_AGENT'] ?? 'unknown'
        ]);
    } catch (Exception $e) {
        error_log("Failed to log security event: " . $e->getMessage());
    }
}

/**
 * Check for suspicious activity
 */
function checkSuspiciousActivity($ip_address) {
    try {
        $pdo = getDB();
        
        // Check for multiple failed login attempts
        $stmt = $pdo->prepare("
            SELECT COUNT(*) as attempts 
            FROM security_logs 
            WHERE event = 'FAILED_LOGIN' 
            AND ip_address = ? 
            AND created_at > DATE_SUB(NOW(), INTERVAL 1 HOUR)
        ");
        $stmt->execute([$ip_address]);
        $result = $stmt->fetch();
        
        if ($result['attempts'] > 10) {
            logSecurityEvent('SUSPICIOUS_ACTIVITY', "Multiple failed logins from IP: $ip_address", 'WARNING');
            return true;
        }
        
        return false;
        
    } catch (Exception $e) {
        error_log("Failed to check suspicious activity: " . $e->getMessage());
        return false;
    }
}

/**
 * Format bytes for display
 */
function formatBytes($bytes, $precision = 2) {
    $units = ['B', 'KB', 'MB', 'GB', 'TB'];
    
    for ($i = 0; $bytes > 1024 && $i < count($units) - 1; $i++) {
        $bytes /= 1024;
    }
    
    return round($bytes, $precision) . ' ' . $units[$i];
}