<?php
/**
 * School Context Manager
 * Handles multi-tenant school selection and context
 */

if (!function_exists('getDB')) {
    require_once __DIR__ . '/../config.php';
}

/**
 * Get the current school's reference code from session
 * @return string|null Academy reference code (e.g., 'KINE')
 */
function getCurrentSchool() {
    return $_SESSION['academy_reference'] ?? null;
}

/**
 * Get the current school's full details
 * @return array|null School details array
 */
function getCurrentSchoolData() {
    $ref = getCurrentSchool();
    if (!$ref) {
        return null;
    }
    
    // Check cache first
    if (isset($_SESSION['school_data_cache']) && 
        $_SESSION['school_data_cache']['reference_code'] === $ref) {
        return $_SESSION['school_data_cache'];
    }
    
    $pdo = getDB();
    $stmt = $pdo->prepare("
        SELECT * FROM academy_references 
        WHERE reference_code = ? AND is_active = 1
    ");
    $stmt->execute([$ref]);
    $school = $stmt->fetch();
    
    if ($school) {
        $_SESSION['school_data_cache'] = $school;
    }
    
    return $school;
}

/**
 * Set the current school context
 * @param string $reference_code Academy reference code
 * @return bool Success status
 */
function setCurrentSchool($reference_code) {
    $pdo = getDB();
    
    try {
        $stmt = $pdo->prepare("
            SELECT * FROM academy_references 
            WHERE reference_code = ? AND is_active = 1
        ");
        $stmt->execute([$reference_code]);
        $school = $stmt->fetch();
        
        if ($school) {
            $_SESSION['academy_reference'] = $reference_code;
            $_SESSION['academy_name'] = $school['academy_name'];
            $_SESSION['academy_id'] = $school['id'];
            $_SESSION['admin_id'] = $school['admin_id'];
            $_SESSION['school_data_cache'] = $school;
            
            // Log school selection
            logSchoolActivity('SCHOOL_SELECTED', "Switched to {$school['academy_name']}");
            
            return true;
        }
        
        return false;
        
    } catch (PDOException $e) {
        error_log("Error setting school: " . $e->getMessage());
        return false;
    }
}

/**
 * Require a school to be selected - middleware function
 * Redirects to school selection page if no school selected
 */
function requireSchool() {
    // Skip if already on select_school.php to prevent loop
    if (isset($_SERVER['PHP_SELF']) && strpos($_SERVER['PHP_SELF'], 'select_school.php') !== false) {
        return;
    }
    
    if (!getCurrentSchool()) {
        // Check if user is logged in as admin
        if (isset($_SESSION['admin_id'])) {
            // Try to set school from admin's academy_reference
            $pdo = getDB();
            $stmt = $pdo->prepare("SELECT academy_reference FROM admins WHERE id = ?");
            $stmt->execute([$_SESSION['admin_id']]);
            $admin_ref = $stmt->fetchColumn();
            
            if ($admin_ref && setCurrentSchool($admin_ref)) {
                return; // School set successfully
            }
        }
        
        // Determine correct path to select_school.php
        $currentPath = $_SERVER['PHP_SELF'] ?? '';
        if (strpos($currentPath, '/admin/') !== false) {
            $redirectPath = '../select_school.php';
        } elseif (strpos($currentPath, '/super_admin/') !== false) {
            $redirectPath = '../select_school.php';
        } else {
            $redirectPath = 'select_school.php';
        }
        
        // Redirect to school selection
        header('Location: ' . $redirectPath);
        exit;
    }
}

/**
 * Check if user is super admin (can access all schools)
 * @return bool
 */
function isSuperAdmin() {
    return isset($_SESSION['is_super_admin']) && $_SESSION['is_super_admin'] === true;
}

/**
 * Validate that admin has access to specified school
 * @param string $academy_ref Academy reference to check
 * @return bool
 */
function validateSchoolAccess($academy_ref) {
    // Super admin can access everything
    if (isSuperAdmin()) {
        return true;
    }
    
    // Regular admin can only access their own school
    $current = getCurrentSchool();
    return $current === $academy_ref;
}

/**
 * Get a school-specific setting value
 * @param string $key Setting key
 * @param mixed $default Default value if not found
 * @return string|null Setting value
 */
function getSchoolSetting($key, $default = null) {
    $academy_ref = getCurrentSchool();
    if (!$academy_ref) {
        return $default;
    }
    
    // Check cache first
    $cacheKey = "school_setting_{$academy_ref}_{$key}";
    if (isset($_SESSION[$cacheKey])) {
        return $_SESSION[$cacheKey];
    }
    
    $pdo = getDB();
    try {
        $stmt = $pdo->prepare("
            SELECT setting_value FROM tenant_settings 
            WHERE academy_reference = ? AND setting_key = ?
        ");
        $stmt->execute([$academy_ref, $key]);
        $value = $stmt->fetchColumn();
        
        $result = $value ?: $default;
        $_SESSION[$cacheKey] = $result; // Cache it
        
        return $result;
    } catch (PDOException $e) {
        error_log("Error getting school setting: " . $e->getMessage());
        return $default;
    }
}

/**
 * Set a school-specific setting value
 * @param string $key Setting key
 * @param string $value Setting value
 * @return bool Success status
 */
function setSchoolSetting($key, $value) {
    $academy_ref = getCurrentSchool();
    if (!$academy_ref) {
        return false;
    }
    
    $pdo = getDB();
    try {
        $stmt = $pdo->prepare("
            INSERT INTO tenant_settings (academy_reference, setting_key, setting_value)
            VALUES (?, ?, ?)
            ON DUPLICATE KEY UPDATE setting_value = ?, updated_at = CURRENT_TIMESTAMP
        ");
        $stmt->execute([$academy_ref, $key, $value, $value]);
        
        // Clear cache
        $cacheKey = "school_setting_{$academy_ref}_{$key}";
        unset($_SESSION[$cacheKey]);
        
        return true;
    } catch (PDOException $e) {
        error_log("Error setting school setting: " . $e->getMessage());
        return false;
    }
}

/**
 * Get all active schools
 * @return array Array of school data
 */
function getAllSchools() {
    $pdo = getDB();
    
    try {
        $stmt = $pdo->query("
            SELECT * FROM academy_references 
            WHERE is_active = 1 
            ORDER BY academy_name ASC
        ");
        return $stmt->fetchAll();
    } catch (PDOException $e) {
        error_log("Error getting schools: " . $e->getMessage());
        return [];
    }
}

/**
 * Get school statistics
 * @param string $academy_ref Academy reference code
 * @return array Statistics array
 */
function getSchoolStats($academy_ref = null) {
    $academy_ref = $academy_ref ?? getCurrentSchool();
    if (!$academy_ref) {
        return [];
    }
    
    $pdo = getDB();
    
    try {
        // Get student count
        $stmt = $pdo->prepare("
            SELECT COUNT(*) FROM students 
            WHERE academy_reference = ? AND deleted_at IS NULL
        ");
        $stmt->execute([$academy_ref]);
        $studentCount = $stmt->fetchColumn();
        
        // Get payment stats
        $stmt = $pdo->prepare("
            SELECT 
                COUNT(*) as total_payments,
                SUM(amount) as total_revenue,
                SUM(admin_earnings) as total_earnings
            FROM payments 
            WHERE academy_reference = ? AND status = 'SUCCESSFUL'
        ");
        $stmt->execute([$academy_ref]);
        $paymentStats = $stmt->fetch();
        
        // Get active student count
        $stmt = $pdo->prepare("
            SELECT COUNT(*) FROM students 
            WHERE academy_reference = ? 
            AND status = 'active' 
            AND deleted_at IS NULL
        ");
        $stmt->execute([$academy_ref]);
        $activeStudents = $stmt->fetchColumn();
        
        return [
            'total_students' => $studentCount,
            'active_students' => $activeStudents,
            'total_payments' => $paymentStats['total_payments'] ?? 0,
            'total_revenue' => $paymentStats['total_revenue'] ?? 0.00,
            'total_earnings' => $paymentStats['total_earnings'] ?? 0.00,
            'academy_reference' => $academy_ref
        ];
        
    } catch (PDOException $e) {
        error_log("Error getting school stats: " . $e->getMessage());
        return [];
    }
}

/**
 * Log school-related activity
 * @param string $action Action type
 * @param string $details Action details
 */
function logSchoolActivity($action, $details) {
    // If monitoring system exists, use it
    if (function_exists('logUserActivity')) {
        logUserActivity($action, $details);
    }
}

/**
 * Get school display name
 * @return string School name or default
 */
function getSchoolName() {
    $school = getCurrentSchoolData();
    return $school['academy_name'] ?? getSchoolSetting('school_name', 'Skolo-Kine');
}

/**
 * Get school color scheme
 * @return array Colors array
 */
function getSchoolColors() {
    return [
        'primary' => getSchoolSetting('primary_color', '#4F46E5'),
        'secondary' => getSchoolSetting('secondary_color', '#10B981')
    ];
}

/**
 * Clear school context (logout)
 */
function clearSchoolContext() {
    unset($_SESSION['academy_reference']);
    unset($_SESSION['academy_name']);
    unset($_SESSION['academy_id']);
    unset($_SESSION['school_data_cache']);
    
    // Clear all cached settings
    foreach ($_SESSION as $key => $value) {
        if (strpos($key, 'school_setting_') === 0) {
            unset($_SESSION[$key]);
        }
    }
}