<?php
/**
 * Comprehensive Monitoring System for Skolo-Kine
 */

class MonitoringSystem {
    private $pdo;
    private $start_time;
    private $start_memory;
    
    public function __construct() {
        $this->pdo = getDB();
        $this->start_time = microtime(true);
        $this->start_memory = memory_get_usage();
    }
    
    /**
     * Log system metrics
     */
    public function logSystemMetrics() {
        try {
            $metrics = [
                'cpu_usage' => $this->getCPUUsage(),
                'memory_usage' => $this->getMemoryUsage(),
                'disk_usage' => $this->getDiskUsage(),
                'database_connections' => $this->getDatabaseConnections(),
                'active_sessions' => $this->getActiveSessions()
            ];
            
            foreach ($metrics as $name => $value) {
                $this->pdo->prepare("
                    INSERT INTO system_metrics (metric_name, metric_value, metric_unit, server_info, created_at) 
                    VALUES (?, ?, ?, ?, NOW())
                ")->execute([
                    $name,
                    $value['value'],
                    $value['unit'],
                    json_encode($value['info'] ?? [])
                ]);
            }
            
        } catch (Exception $e) {
            error_log("Failed to log system metrics: " . $e->getMessage());
        }
    }
    
    /**
     * Log application performance
     */
    public function logAppPerformance($page_name, $response_time = null, $memory_usage = null, $query_count = 0, $error_count = 0) {
        try {
            $response_time = $response_time ?? (microtime(true) - $this->start_time) * 1000;
            $memory_usage = $memory_usage ?? memory_get_usage() - $this->start_memory;
            
            $this->pdo->prepare("
                INSERT INTO app_performance (page_name, response_time, memory_usage, query_count, error_count, user_agent, ip_address, created_at) 
                VALUES (?, ?, ?, ?, ?, ?, ?, NOW())
            ")->execute([
                $page_name,
                $response_time,
                $memory_usage,
                $query_count,
                $error_count,
                $_SERVER['HTTP_USER_AGENT'] ?? null,
                $_SERVER['REMOTE_ADDR'] ?? null
            ]);
            
        } catch (Exception $e) {
            error_log("Failed to log app performance: " . $e->getMessage());
        }
    }
    
    /**
     * Log database query performance
     */
    public function logDatabaseQuery($query_type, $execution_time, $rows_affected = 0, $query_text = null) {
        try {
            $this->pdo->prepare("
                INSERT INTO database_metrics (query_type, execution_time, rows_affected, query_text, created_at) 
                VALUES (?, ?, ?, ?, NOW())
            ")->execute([
                $query_type,
                $execution_time,
                $rows_affected,
                $query_text
            ]);
            
        } catch (Exception $e) {
            error_log("Failed to log database query: " . $e->getMessage());
        }
    }
    
    /**
     * Log user activity
     */
    public function logUserActivity($user_id, $user_type, $action, $page_url = null, $session_id = null) {
        try {
            $this->pdo->prepare("
                INSERT INTO user_activity (user_id, user_type, action, page_url, ip_address, user_agent, session_id, created_at) 
                VALUES (?, ?, ?, ?, ?, ?, ?, NOW())
            ")->execute([
                $user_id,
                $user_type,
                $action,
                $page_url,
                $_SERVER['REMOTE_ADDR'] ?? null,
                $_SERVER['HTTP_USER_AGENT'] ?? null,
                $session_id
            ]);
            
        } catch (Exception $e) {
            error_log("Failed to log user activity: " . $e->getMessage());
        }
    }
    
    /**
     * Log errors
     */
    public function logError($error_type, $error_message, $error_file = null, $error_line = null, $stack_trace = null, $user_id = null) {
        try {
            $this->pdo->prepare("
                INSERT INTO error_logs (error_type, error_message, error_file, error_line, stack_trace, user_id, ip_address, user_agent, created_at) 
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, NOW())
            ")->execute([
                $error_type,
                $error_message,
                $error_file,
                $error_line,
                $stack_trace,
                $user_id,
                $_SERVER['REMOTE_ADDR'] ?? null,
                $_SERVER['HTTP_USER_AGENT'] ?? null
            ]);
            
        } catch (Exception $e) {
            error_log("Failed to log error: " . $e->getMessage());
        }
    }
    
    /**
     * Log API calls
     */
    public function logAPICall($endpoint, $method, $response_code, $response_time, $request_size = 0, $response_size = 0, $user_id = null) {
        try {
            $this->pdo->prepare("
                INSERT INTO api_monitoring (api_endpoint, method, response_code, response_time, request_size, response_size, user_id, ip_address, created_at) 
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, NOW())
            ")->execute([
                $endpoint,
                $method,
                $response_code,
                $response_time,
                $request_size,
                $response_size,
                $user_id,
                $_SERVER['REMOTE_ADDR'] ?? null
            ]);
            
        } catch (Exception $e) {
            error_log("Failed to log API call: " . $e->getMessage());
        }
    }
    
    /**
     * Log bot activity
     */
    public function logBotActivity($bot_name, $status, $execution_time = null, $records_processed = 0, $errors_count = 0, $memory_usage = null, $cpu_usage = null) {
        try {
            $this->pdo->prepare("
                INSERT INTO bot_monitoring (bot_name, status, last_run, execution_time, records_processed, errors_count, memory_usage, cpu_usage, created_at) 
                VALUES (?, ?, NOW(), ?, ?, ?, ?, ?, NOW())
                ON DUPLICATE KEY UPDATE 
                status = VALUES(status),
                last_run = VALUES(last_run),
                execution_time = VALUES(execution_time),
                records_processed = VALUES(records_processed),
                errors_count = VALUES(errors_count),
                memory_usage = VALUES(memory_usage),
                cpu_usage = VALUES(cpu_usage),
                updated_at = NOW()
            ")->execute([
                $bot_name,
                $status,
                $execution_time,
                $records_processed,
                $errors_count,
                $memory_usage,
                $cpu_usage
            ]);
            
        } catch (Exception $e) {
            error_log("Failed to log bot activity: " . $e->getMessage());
        }
    }
    
    /**
     * Perform health checks
     */
    public function performHealthChecks() {
        $checks = [
            'database_connection' => $this->checkDatabaseConnection(),
            'disk_space' => $this->checkDiskSpace(),
            'memory_usage' => $this->checkMemoryUsage(),
            'database_performance' => $this->checkDatabasePerformance(),
            'email_system' => $this->checkEmailSystem(),
            'moodle_api' => $this->checkMoodleAPI()
        ];
        
        foreach ($checks as $check_name => $result) {
            $this->pdo->prepare("
                INSERT INTO health_checks (check_name, status, check_value, threshold_warning, threshold_critical, message, created_at) 
                VALUES (?, ?, ?, ?, ?, ?, NOW())
            ")->execute([
                $check_name,
                $result['status'],
                $result['value'],
                $result['warning_threshold'],
                $result['critical_threshold'],
                $result['message']
            ]);
            
            // Create alert if critical
            if ($result['status'] === 'CRITICAL') {
                $this->createAlert('SYSTEM_HEALTH', 'CRITICAL', "Health Check Failed: $check_name", $result['message']);
            }
        }
    }
    
    /**
     * Create alert
     */
    public function createAlert($alert_type, $severity, $title, $message) {
        try {
            $this->pdo->prepare("
                INSERT INTO alerts (alert_type, severity, title, message, created_at) 
                VALUES (?, ?, ?, ?, NOW())
            ")->execute([
                $alert_type,
                $severity,
                $title,
                $message
            ]);
            
        } catch (Exception $e) {
            error_log("Failed to create alert: " . $e->getMessage());
        }
    }
    
    /**
     * Get system statistics
     */
    public function getSystemStatistics($hours = 24) {
        try {
            $stats = [];
            
            // System metrics
            $stmt = $this->pdo->prepare("
                SELECT metric_name, AVG(metric_value) as avg_value, MAX(metric_value) as max_value, MIN(metric_value) as min_value
                FROM system_metrics 
                WHERE created_at > DATE_SUB(NOW(), INTERVAL ? HOUR)
                GROUP BY metric_name
            ");
            $stmt->execute([$hours]);
            $stats['system_metrics'] = $stmt->fetchAll();
            
            // App performance
            $stmt = $this->pdo->prepare("
                SELECT page_name, AVG(response_time) as avg_response_time, AVG(memory_usage) as avg_memory
                FROM app_performance 
                WHERE created_at > DATE_SUB(NOW(), INTERVAL ? HOUR)
                GROUP BY page_name
            ");
            $stmt->execute([$hours]);
            $stats['app_performance'] = $stmt->fetchAll();
            
            // Error count
            $stmt = $this->pdo->prepare("
                SELECT COUNT(*) as error_count 
                FROM error_logs 
                WHERE created_at > DATE_SUB(NOW(), INTERVAL ? HOUR)
            ");
            $stmt->execute([$hours]);
            $stats['error_count'] = $stmt->fetch()['error_count'];
            
            // Active alerts
            $stmt = $this->pdo->query("
                SELECT COUNT(*) as active_alerts 
                FROM alerts 
                WHERE status = 'ACTIVE'
            ");
            $stats['active_alerts'] = $stmt->fetch()['active_alerts'];
            
            return $stats;
            
        } catch (Exception $e) {
            error_log("Failed to get system statistics: " . $e->getMessage());
            return [];
        }
    }
    
    // Helper methods for health checks
    private function checkDatabaseConnection() {
        try {
            $start = microtime(true);
            $this->pdo->query("SELECT 1");
            $response_time = (microtime(true) - $start) * 1000;
            
            return [
                'status' => 'HEALTHY',
                'value' => $response_time,
                'warning_threshold' => 1000,
                'critical_threshold' => 5000,
                'message' => "Database connection OK ({$response_time}ms)"
            ];
        } catch (Exception $e) {
            return [
                'status' => 'CRITICAL',
                'value' => 0,
                'warning_threshold' => 1000,
                'critical_threshold' => 5000,
                'message' => "Database connection failed: " . $e->getMessage()
            ];
        }
    }
    
    private function checkDiskSpace() {
        $free_bytes = disk_free_space(__DIR__);
        $total_bytes = disk_total_space(__DIR__);
        $used_percent = (($total_bytes - $free_bytes) / $total_bytes) * 100;
        
        $status = 'HEALTHY';
        if ($used_percent > 90) $status = 'CRITICAL';
        elseif ($used_percent > 80) $status = 'WARNING';
        
        return [
            'status' => $status,
            'value' => $used_percent,
            'warning_threshold' => 80,
            'critical_threshold' => 90,
            'message' => "Disk usage: " . round($used_percent, 2) . "%"
        ];
    }
    
    private function checkMemoryUsage() {
        $memory_usage = memory_get_usage(true);
        $memory_limit = ini_get('memory_limit');
        $memory_limit_bytes = $this->parseMemoryLimit($memory_limit);
        $usage_percent = ($memory_usage / $memory_limit_bytes) * 100;
        
        $status = 'HEALTHY';
        if ($usage_percent > 90) $status = 'CRITICAL';
        elseif ($usage_percent > 80) $status = 'WARNING';
        
        return [
            'status' => $status,
            'value' => $usage_percent,
            'warning_threshold' => 80,
            'critical_threshold' => 90,
            'message' => "Memory usage: " . round($usage_percent, 2) . "%"
        ];
    }
    
    private function checkDatabasePerformance() {
        try {
            $start = microtime(true);
            $this->pdo->query("SELECT COUNT(*) FROM students");
            $response_time = (microtime(true) - $start) * 1000;
            
            $status = 'HEALTHY';
            if ($response_time > 2000) $status = 'CRITICAL';
            elseif ($response_time > 1000) $status = 'WARNING';
            
            return [
                'status' => $status,
                'value' => $response_time,
                'warning_threshold' => 1000,
                'critical_threshold' => 2000,
                'message' => "Database query time: {$response_time}ms"
            ];
        } catch (Exception $e) {
            return [
                'status' => 'CRITICAL',
                'value' => 0,
                'warning_threshold' => 1000,
                'critical_threshold' => 2000,
                'message' => "Database performance check failed: " . $e->getMessage()
            ];
        }
    }
    
    private function checkEmailSystem() {
        try {
            require_once __DIR__ . '/email_functions.php';
            $result = testEmailConfiguration();
            
            return [
                'status' => $result['success'] ? 'HEALTHY' : 'CRITICAL',
                'value' => $result['success'] ? 1 : 0,
                'warning_threshold' => 0.5,
                'critical_threshold' => 0,
                'message' => $result['message']
            ];
        } catch (Exception $e) {
            return [
                'status' => 'CRITICAL',
                'value' => 0,
                'warning_threshold' => 0.5,
                'critical_threshold' => 0,
                'message' => "Email system check failed: " . $e->getMessage()
            ];
        }
    }
    
    private function checkMoodleAPI() {
        try {
            $api_url = MOODLE_API_URL;
            $params = [
                'wstoken' => MOODLE_API_TOKEN,
                'wsfunction' => 'core_webservice_get_site_info',
                'moodlewsrestformat' => 'json'
            ];
            
            $ch = curl_init();
            curl_setopt($ch, CURLOPT_URL, $api_url);
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($params));
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_TIMEOUT, 10);
            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
            
            $start = microtime(true);
            $response = curl_exec($ch);
            $response_time = (microtime(true) - $start) * 1000;
            $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            curl_close($ch);
            
            $status = 'HEALTHY';
            if ($http_code !== 200) $status = 'CRITICAL';
            elseif ($response_time > 5000) $status = 'WARNING';
            
            return [
                'status' => $status,
                'value' => $response_time,
                'warning_threshold' => 5000,
                'critical_threshold' => 10000,
                'message' => "Moodle API: HTTP $http_code, {$response_time}ms"
            ];
        } catch (Exception $e) {
            return [
                'status' => 'CRITICAL',
                'value' => 0,
                'warning_threshold' => 5000,
                'critical_threshold' => 10000,
                'message' => "Moodle API check failed: " . $e->getMessage()
            ];
        }
    }
    
    // Helper methods for system metrics
    private function getCPUUsage() {
        // This is a simplified version - in production, you'd use more sophisticated methods
        if (function_exists('sys_getloadavg')) {
            $load = sys_getloadavg();
            return [
                'value' => $load[0] * 100,
                'unit' => 'percent',
                'info' => ['load_average' => $load]
            ];
        } else {
            // Windows fallback
            return [
                'value' => 0,
                'unit' => 'percent',
                'info' => ['load_average' => 'Not available on Windows']
            ];
        }
    }
    
    private function getMemoryUsage() {
        $usage = memory_get_usage(true);
        $peak = memory_get_peak_usage(true);
        
        return [
            'value' => $usage,
            'unit' => 'bytes',
            'info' => [
                'current' => $usage,
                'peak' => $peak,
                'limit' => ini_get('memory_limit')
            ]
        ];
    }
    
    private function getDiskUsage() {
        $free = disk_free_space(__DIR__);
        $total = disk_total_space(__DIR__);
        $used = $total - $free;
        
        return [
            'value' => ($used / $total) * 100,
            'unit' => 'percent',
            'info' => [
                'free' => $free,
                'total' => $total,
                'used' => $used
            ]
        ];
    }
    
    private function getDatabaseConnections() {
        try {
            $stmt = $this->pdo->query("SHOW STATUS LIKE 'Threads_connected'");
            $result = $stmt->fetch();
            return [
                'value' => $result['Value'] ?? 0,
                'unit' => 'connections',
                'info' => []
            ];
        } catch (Exception $e) {
            return [
                'value' => 0,
                'unit' => 'connections',
                'info' => ['error' => $e->getMessage()]
            ];
        }
    }
    
    private function getActiveSessions() {
        try {
            $stmt = $this->pdo->query("SELECT COUNT(*) as count FROM user_sessions WHERE is_active = 1 AND expires_at > NOW()");
            $result = $stmt->fetch();
            return [
                'value' => $result['count'],
                'unit' => 'sessions',
                'info' => []
            ];
        } catch (Exception $e) {
            return [
                'value' => 0,
                'unit' => 'sessions',
                'info' => ['error' => $e->getMessage()]
            ];
        }
    }
    
    private function parseMemoryLimit($limit) {
        $limit = trim($limit);
        $last = strtolower($limit[strlen($limit)-1]);
        $limit = (int) $limit;
        
        switch($last) {
            case 'g': $limit *= 1024;
            case 'm': $limit *= 1024;
            case 'k': $limit *= 1024;
        }
        
        return $limit;
    }
}

// Auto-initialize monitoring if not already done
if (!defined('MONITORING_INITIALIZED')) {
    define('MONITORING_INITIALIZED', true);
    
    // Initialize monitoring
    $monitoring = new MonitoringSystem();
    
    // Log system metrics every 5 minutes
    if (rand(1, 100) <= 20) { // 20% chance to run (simulating cron)
        $monitoring->logSystemMetrics();
    }
}